// Package priorityqueue provides a generic priority queue with Insert,
// PopMax, and RemoveFunc operations.
package priorityqueue

// PriorityQueue is a generic priority queue with a configurable comparison
// function.
type PriorityQueue[T any] struct {
	cmp func(a, b T) int

	// items holds the queue's items as a binary heap.
	// items[0] is a dummy element that's not used. If the queue has N elements,
	// they are stored at indices 1...N (N == len(items)-1)
	// For an element at index i, its parent is at index i/2, and its children
	// are at indices 2i and 2i+1. The root of the heap is at index 1.
	items []T
}

// New creates a new PriorityQueue, configured with a function that
// compares the priorities of two items a and b; it should return a number > 0
// if the priority of a is higher, 0 if the priorities are equal, and a
// number < 0 otherwise.
// sizeHint sets the initial capacity of the queue; -1 means to use the default.
func New[T any](sizeHint int, cmp func(a, b T) int) *PriorityQueue[T] {
	return &PriorityQueue[T]{cmp: cmp, items: make([]T, 1, max(1, sizeHint+1))}
}

// Len returns the length (number of items) of the priority queue.
func (pq *PriorityQueue[T]) Len() int {
	return len(pq.items) - 1
}

// Insert inserts a new element into the priority queue.
func (pq *PriorityQueue[T]) Insert(elem T) {
	pq.items = append(pq.items, elem)
	pq.siftup(len(pq.items) - 1)
}

// PopMax returns the element with the maximal priority in the queue, and
// removes it from the queue. Warning: to maintain a clean API, PopMax panics
// if the queue is empty. Make sure to check Len() first.
func (pq *PriorityQueue[T]) PopMax() T {
	if len(pq.items) < 2 {
		panic("popping from empty priority queue")
	}
	maxItem := pq.items[1]
	pq.items[1] = pq.items[len(pq.items)-1]
	pq.items = pq.items[:len(pq.items)-1]
	pq.siftdown(1)
	return maxItem
}

// RemoveFunc removes all elements for which rm returns true.
func (pq *PriorityQueue[T]) RemoveFunc(rm func(T) bool) {
	// This is effectively slices.DeleteFunc, but inlined because we start from index 1.
	i := 1
	for ; i < len(pq.items); i++ {
		if rm(pq.items[i]) {
			break
		}
	}
	if i == len(pq.items) {
		return // nothing to remove
	}
	for j := i + 1; j < len(pq.items); j++ {
		if v := pq.items[j]; !rm(v) {
			pq.items[i] = v
			i++
		}
	}
	// Clear the tail.
	clear(pq.items[i:])
	pq.items = pq.items[:i]
	pq.rebuildHeap()
}

// rebuildHeap rebuilds the entire heap from scratch.
func (pq *PriorityQueue[T]) rebuildHeap() {
	for i := len(pq.items) / 2; i >= 1; i-- {
		pq.siftdown(i)
	}
}

func (pq *PriorityQueue[T]) siftup(n int) {
	i := n
	for {
		if i == 1 {
			// Reached root, we're done.
			return
		}
		// p is the index of i's parent
		// if p parent has a higher priority than i, we're done.
		p := i / 2
		if pq.cmp(pq.items[p], pq.items[i]) >= 0 {
			return
		}
		pq.items[i], pq.items[p] = pq.items[p], pq.items[i]
		i = p
	}
}

func (pq *PriorityQueue[T]) siftdown(i int) {
	for {
		c := 2 * i
		if c >= len(pq.items) {
			return
		}
		// c is not out of bounds, so it's the index of the left child of i

		// Figure out the child index with the maximal priority
		maxChild := c
		if c+1 < len(pq.items) {
			// c+1 is not out of bounds, so it's the index of the right child of i
			if pq.cmp(pq.items[c+1], pq.items[c]) > 0 {
				maxChild = c + 1
			}
		}
		if pq.cmp(pq.items[i], pq.items[maxChild]) >= 0 {
			// i has higher priority than either child, so we're done.
			return
		}

		pq.items[i], pq.items[maxChild] = pq.items[maxChild], pq.items[i]
		i = maxChild
	}
}
