package test_test

import (
	"encoding/csv"
	"encoding/json"
	"errors"
	"os"
	"path/filepath"
	"strings"
	"testing"
	"time"

	"github.com/gruntwork-io/terragrunt/test/helpers"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

const (
	testFixtureFilterGraphDAG = "fixtures/find/dag"
	testFixtureRunFilter      = "fixtures/run-filter"
)

func TestFilterFlagWithFindGraphExpressions(t *testing.T) {
	t.Parallel()

	// Skip if filter-flag experiment is not enabled
	if !helpers.IsExperimentMode(t) {
		t.Skip("Skipping filter flag tests - TG_EXPERIMENT_MODE not enabled")
	}

	testCases := []struct {
		name           string
		filterQuery    string
		expectedOutput string
		expectError    bool
	}{
		{
			// a-dependent -> b-dependency
			// So "a-dependent..." should find a-dependent and b-dependency
			name:           "dependency traversal - a-dependent...",
			filterQuery:    "a-dependent...",
			expectedOutput: "a-dependent\nb-dependency\n",
			expectError:    false,
		},
		{
			// b-dependency is a dependency of a-dependent, c-mixed-deps, and d-dependencies-only
			// So "...b-dependency" should find b-dependency and all its dependents
			// Note: Actually, b-dependency has no dependents in this graph - it's only a dependency
			// But c-mixed-deps depends on a-dependent which depends on b-dependency
			// And d-dependencies-only depends on a-dependent which depends on b-dependency
			// So ...b-dependency should find: b-dependency, a-dependent, c-mixed-deps, d-dependencies-only
			name:           "dependent traversal - ...b-dependency",
			filterQuery:    "...b-dependency",
			expectedOutput: "a-dependent\nb-dependency\nc-mixed-deps\nd-dependencies-only\n",
			expectError:    false,
		},
		{
			// a-dependent has dependencies (b-dependency) and dependents (c-mixed-deps, d-dependencies-only)
			// So "...a-dependent..." should find all: b-dependency, a-dependent, c-mixed-deps, d-dependencies-only
			name:           "both directions - ...a-dependent...",
			filterQuery:    "...a-dependent...",
			expectedOutput: "a-dependent\nb-dependency\nc-mixed-deps\nd-dependencies-only\n",
			expectError:    false,
		},
		{
			// "a-dependent..." finds a-dependent and b-dependency
			// "^a-dependent..." excludes a-dependent, so only b-dependency
			name:           "exclude target - ^a-dependent...",
			filterQuery:    "^a-dependent...",
			expectedOutput: "b-dependency\n",
			expectError:    false,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			t.Parallel()

			helpers.CleanupTerraformFolder(t, testFixtureFilterGraphDAG)
			tmpEnvPath := helpers.CopyEnvironment(t, testFixtureFilterGraphDAG)
			workingDir := filepath.Join(tmpEnvPath, testFixtureFilterGraphDAG)

			cmd := "terragrunt find --no-color --working-dir " + workingDir + " --filter '" + tc.filterQuery + "'"
			stdout, stderr, err := helpers.RunTerragruntCommandWithOutput(t, cmd)

			if tc.expectError {
				require.Error(t, err, "Expected error for filter query: %s", tc.filterQuery)
				assert.NotEmpty(t, stderr, "Expected error message in stderr")
			} else {
				require.NoError(t, err, "Unexpected error for filter query: %s", tc.filterQuery)
				// Allow warnings in stderr (e.g., suppressed parsing errors during discovery)
				// but ensure there are no actual errors
				if stderr != "" {
					// Check that stderr only contains expected warnings, not actual errors
					lowerStderr := strings.ToLower(stderr)
					if strings.Contains(lowerStderr, "error") && !strings.Contains(lowerStderr, "suppressed") && !strings.Contains(lowerStderr, "warning") {
						t.Errorf("Unexpected error in stderr: %s", stderr)
					}
				}

				// Sort both outputs for comparison (find output order may vary)
				expectedLines := strings.Fields(tc.expectedOutput)
				actualLines := strings.Fields(stdout)
				assert.ElementsMatch(t, expectedLines, actualLines, "Output mismatch for filter query: %s", tc.filterQuery)
			}
		})
	}
}

func TestFilterFlagWithFindGraphExpressionsJSON(t *testing.T) {
	t.Parallel()

	// Skip if filter-flag experiment is not enabled
	if !helpers.IsExperimentMode(t) {
		t.Skip("Skipping filter flag tests - TG_EXPERIMENT_MODE not enabled")
	}

	testCases := []struct {
		name          string
		filterQuery   string
		expectedPaths []string
		expectError   bool
	}{
		{
			name:          "dependency traversal - a-dependent... JSON",
			filterQuery:   "a-dependent...",
			expectedPaths: []string{"a-dependent", "b-dependency"},
			expectError:   false,
		},
		{
			name:          "dependent traversal - ...b-dependency JSON",
			filterQuery:   "...b-dependency",
			expectedPaths: []string{"a-dependent", "b-dependency", "c-mixed-deps", "d-dependencies-only"},
			expectError:   false,
		},
		{
			name:          "both directions - ...a-dependent... JSON",
			filterQuery:   "...a-dependent...",
			expectedPaths: []string{"a-dependent", "b-dependency", "c-mixed-deps", "d-dependencies-only"},
			expectError:   false,
		},
		{
			name:          "exclude target - ^a-dependent... JSON",
			filterQuery:   "^a-dependent...",
			expectedPaths: []string{"b-dependency"},
			expectError:   false,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			t.Parallel()

			helpers.CleanupTerraformFolder(t, testFixtureFilterGraphDAG)
			tmpEnvPath := helpers.CopyEnvironment(t, testFixtureFilterGraphDAG)
			workingDir := filepath.Join(tmpEnvPath, testFixtureFilterGraphDAG)

			cmd := "terragrunt find --no-color --working-dir " + workingDir + " --json --filter '" + tc.filterQuery + "'"
			stdout, stderr, err := helpers.RunTerragruntCommandWithOutput(t, cmd)

			if tc.expectError {
				require.Error(t, err, "Expected error for filter query: %s", tc.filterQuery)
				assert.NotEmpty(t, stderr, "Expected error message in stderr")

				return
			}

			require.NoError(t, err, "Unexpected error for filter query: %s", tc.filterQuery)

			// Parse JSON output and verify paths
			// The JSON output should be an array of objects with "path" field
			assert.NotEmpty(t, stdout, "JSON output should not be empty")
			assert.Contains(t, stdout, "[", "JSON output should be an array")

			// Verify each expected path appears in the JSON output
			for _, expectedPath := range tc.expectedPaths {
				assert.Contains(t, stdout, `"path"`, "JSON output should contain path field")
				// The path might be relative or absolute, so we check for the component name
				assert.Contains(t, stdout, expectedPath, "JSON output should contain path: %s", expectedPath)
			}
		})
	}
}

func TestFilterFlagWithRunGraphExpressions(t *testing.T) {
	t.Parallel()

	// Skip if filter-flag experiment is not enabled
	if !helpers.IsExperimentMode(t) {
		t.Skip("Skipping filter flag tests - TG_EXPERIMENT_MODE not enabled")
	}

	testCases := []struct {
		name         string
		filterQuery  string
		errorPattern string
		expectError  bool
	}{
		{
			name:        "dependency traversal - a-dependent...",
			filterQuery: "a-dependent...",
			expectError: false,
		},
		{
			name:        "dependent traversal - ...b-dependency",
			filterQuery: "...b-dependency",
			expectError: false,
		},
		{
			name:        "both directions - ...a-dependent...",
			filterQuery: "...a-dependent...",
			expectError: false,
		},
		{
			name:        "exclude target - ^a-dependent...",
			filterQuery: "^a-dependent...",
			expectError: false,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			t.Parallel()

			helpers.CleanupTerraformFolder(t, testFixtureFilterGraphDAG)
			tmpEnvPath := helpers.CopyEnvironment(t, testFixtureFilterGraphDAG)
			workingDir := filepath.Join(tmpEnvPath, testFixtureFilterGraphDAG)

			// Use a non-destructive command like `plan` to verify the filter works
			// The actual terraform commands will likely fail due to missing providers/resources,
			// but we can verify that the filter parsing and discovery works correctly
			// by checking that we don't get filter-related errors
			cmd := "terragrunt run --all --non-interactive --working-dir " + workingDir + " --filter '" + tc.filterQuery + "' -- plan"
			stdout, stderr, err := helpers.RunTerragruntCommandWithOutput(t, cmd)

			if tc.expectError {
				require.Error(t, err, "Expected error for filter query: %s", tc.filterQuery)

				if tc.errorPattern != "" {
					assert.Contains(t, stderr, tc.errorPattern, "Error message should contain expected pattern")
				}
			} else {
				// The command might fail due to terraform init/plan errors (missing providers, etc),
				// which is expected in a test environment without full terraform setup.
				// The important thing is that the filter was parsed correctly and discovery worked.
				output := stdout + stderr

				// Verify we don't get filter parsing or evaluation errors
				errStr := ""
				if err != nil {
					errStr = err.Error()
				}

				// Check for filter-related errors (these would indicate a problem with graph expressions)
				if strings.Contains(output, "filter") {
					if strings.Contains(output, "parse") || strings.Contains(output, "syntax") || strings.Contains(output, "invalid") {
						t.Fatalf("Filter parsing/evaluation error detected in output: %s\nOutput: %s\nStderr: %s", errStr, stdout, stderr)
					}
				}

				// Check error string directly for filter issues
				if err != nil {
					if strings.Contains(errStr, "filter") && (strings.Contains(errStr, "parse") || strings.Contains(errStr, "syntax") || strings.Contains(errStr, "invalid")) {
						t.Fatalf("Filter parsing/evaluation error: %v\nOutput: %s\nStderr: %s", err, stdout, stderr)
					}
					// Terraform execution errors are acceptable - we're just verifying filter discovery works
					t.Logf("Command completed (Terraform execution errors are expected in test environment): %v", err)
				}

				// Verify that the command at least attempted to process units (discovery phase completed)
				// This is a basic sanity check - if discovery failed, we'd see different errors
				assert.NotEmpty(t, output, "Command should produce some output")
			}
		})
	}
}

func TestFilterFlagWithRunAllGraphExpressions(t *testing.T) {
	t.Parallel()

	// Skip if filter-flag experiment is not enabled
	if !helpers.IsExperimentMode(t) {
		t.Skip("Skipping filter flag tests - TG_EXPERIMENT_MODE not enabled")
	}

	testCases := []struct {
		name          string
		filterQuery   string
		expectedUnits []string // Expected units to be executed (based on graph traversal)
		expectError   bool
	}{
		{
			// service -> db, cache, vpc (all dependencies)
			// So "service..." should execute service and all its dependencies
			name:          "dependency traversal - service... executes dependencies",
			filterQuery:   "service...",
			expectedUnits: []string{"service", "db", "cache", "vpc"},
			expectError:   false,
		},
		{
			// vpc has dependents: db, cache, service (all depend on vpc)
			// So "...vpc" should execute all: vpc, db, cache, service
			name:          "dependent traversal - ...vpc executes all dependents",
			filterQuery:   "...vpc",
			expectedUnits: []string{"vpc", "db", "cache", "service"},
			expectError:   false,
		},
		{
			// db has dependency (vpc) and dependent (service)
			// So "...db..." should execute all: vpc, db, service
			name:          "both directions - ...db... executes related units",
			filterQuery:   "...db...",
			expectedUnits: []string{"vpc", "db", "service"},
			expectError:   false,
		},
		{
			// cache has dependency (vpc) and dependent (service)
			// So "...cache..." should execute all: vpc, cache, service
			name:          "both directions - ...cache... executes related units",
			filterQuery:   "...cache...",
			expectedUnits: []string{"vpc", "cache", "service"},
			expectError:   false,
		},
		{
			// "service..." finds service, db, cache, vpc
			// "^service..." excludes service, so only dependencies should execute
			name:          "exclude target - ^service... executes only dependencies",
			filterQuery:   "^service...",
			expectedUnits: []string{"db", "cache", "vpc"},
			expectError:   false,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			t.Parallel()

			helpers.CleanupTerraformFolder(t, testFixtureRunFilter)
			tmpEnvPath := helpers.CopyEnvironment(t, testFixtureRunFilter)
			workingDir := filepath.Join(tmpEnvPath, testFixtureRunFilter)

			// Use a non-destructive command like `plan` to verify the filter works
			// The actual terraform commands will likely fail due to missing providers/resources,
			// but we can verify that the filter correctly selects units by checking the output and report
			reportFile := filepath.Join(workingDir, "report.json")
			cmd := "terragrunt run --all --non-interactive --working-dir " + workingDir + " --filter '" + tc.filterQuery + "' --report-file " + reportFile + " --report-format json -- plan"
			stdout, stderr, err := helpers.RunTerragruntCommandWithOutput(t, cmd)

			if tc.expectError {
				require.Error(t, err, "Expected error for filter query: %s", tc.filterQuery)
			} else {
				output := stdout + stderr

				// Verify we don't get filter parsing or evaluation errors
				errStr := ""
				if err != nil {
					errStr = err.Error()
				}

				// Check for filter-related errors (these would indicate a problem with graph expressions)
				if strings.Contains(output, "filter") {
					if strings.Contains(output, "parse") || strings.Contains(output, "syntax") || strings.Contains(output, "invalid") {
						t.Fatalf("Filter parsing/evaluation error detected in output: %s\nOutput: %s\nStderr: %s", errStr, stdout, stderr)
					}
				}

				// Check error string directly for filter issues
				if err != nil {
					if strings.Contains(errStr, "filter") && (strings.Contains(errStr, "parse") || strings.Contains(errStr, "syntax") || strings.Contains(errStr, "invalid")) {
						t.Fatalf("Filter parsing/evaluation error: %v\nOutput: %s\nStderr: %s", err, stdout, stderr)
					}
					// Terraform execution errors are acceptable - we're just verifying filter discovery works
					t.Logf("Command completed (Terraform execution errors are expected in test environment): %v", err)
				}

				// Verify that the command at least attempted to process units (discovery phase completed)
				assert.NotEmpty(t, output, "Command should produce some output")

				// Verify that expected units appear in the output (they should be discovered and processed)
				// The output should contain references to the units being processed
				// Note: Since terraform may fail, we check for unit names in paths or logs
				for _, expectedUnit := range tc.expectedUnits {
					unitName := expectedUnit
					// Check if the unit appears in the output (might be in paths, logs, or error messages)
					// We're lenient here since terraform execution may fail, but discovery should have happened
					found := strings.Contains(output, unitName)
					if !found {
						t.Logf("Warning: Expected unit '%s' not found in output, but this may be due to terraform execution errors. Output: %s", unitName, output)
					}
				}

				// Verify run report contains expected units
				if _, statErr := os.Stat(reportFile); statErr == nil {
					reportUnits, parseErr := parseReportFile(reportFile)
					if parseErr != nil {
						t.Logf("Warning: Could not parse report file: %v. Skipping report verification.", parseErr)
					} else {
						// Verify all expected units are in the report
						reportUnitMap := make(map[string]bool)
						for _, unit := range reportUnits {
							reportUnitMap[unit] = true
						}

						for _, expectedUnit := range tc.expectedUnits {
							// Check if unit name appears in any report entry (might be full path or just name)
							found := false

							for reportUnit := range reportUnitMap {
								if strings.Contains(reportUnit, expectedUnit) || strings.Contains(expectedUnit, reportUnit) {
									found = true
									break
								}
							}

							if !found {
								t.Logf("Warning: Expected unit '%s' not found in report. Report units: %v", expectedUnit, reportUnits)
							} else {
								t.Logf("Verified unit '%s' found in report", expectedUnit)
							}
						}
					}
				} else {
					t.Logf("Warning: Report file not found at %s. Skipping report verification.", reportFile)
				}
			}
		})
	}
}

func TestFilterFlagWithRunAllGraphExpressionsVerifyExecutionOrder(t *testing.T) {
	t.Parallel()

	// Skip if filter-flag experiment is not enabled
	if !helpers.IsExperimentMode(t) {
		t.Skip("Skipping filter flag tests - TG_EXPERIMENT_MODE not enabled")
	}

	// This test verifies that when using graph expressions, dependencies are executed before dependents
	// We'll use a simple dependency chain to verify execution order
	helpers.CleanupTerraformFolder(t, testFixtureRunFilter)
	tmpEnvPath := helpers.CopyEnvironment(t, testFixtureRunFilter)
	workingDir := filepath.Join(tmpEnvPath, testFixtureRunFilter)

	// Test that "service..." executes vpc, db, cache (dependencies) before service
	reportFile := filepath.Join(workingDir, "report.json")
	cmd := "terragrunt run --all --non-interactive --working-dir " + workingDir + " --filter 'service...' --report-file " + reportFile + " --report-format json -- plan"
	stdout, stderr, err := helpers.RunTerragruntCommandWithOutput(t, cmd)

	output := stdout + stderr

	// Verify no filter errors
	if strings.Contains(output, "filter") && (strings.Contains(output, "parse") || strings.Contains(output, "syntax")) {
		t.Fatalf("Filter parsing error: %v\nOutput: %s\nStderr: %s", err, stdout, stderr)
	}

	// Even if terraform fails, we should verify that all units were discovered
	// The key is that filter parsing and discovery worked correctly
	assert.NotEmpty(t, output, "Command should produce some output")

	// Verify that the filter was processed (no filter-related errors means it worked)
	if err != nil {
		errStr := err.Error()
		if strings.Contains(errStr, "filter") && (strings.Contains(errStr, "parse") || strings.Contains(errStr, "syntax")) {
			t.Fatalf("Filter parsing error: %v", err)
		}
		// Other errors are expected (terraform init/plan failures)
		t.Logf("Terraform execution error (expected): %v", err)
	}

	// Parse report file to verify execution order
	if _, statErr := os.Stat(reportFile); statErr == nil {
		reportRecords, parseErr := parseReportFileWithTimestamps(reportFile)
		if parseErr != nil {
			t.Logf("Warning: Could not parse report file: %v. Skipping execution order verification.", parseErr)
		} else {
			// Verify execution order: dependencies (vpc, db, cache) should start before service
			// We expect: vpc, db, cache should have started before service
			dependencies := []string{"vpc", "db", "cache"}
			dependent := "service"

			// Find service start time
			var serviceStartTime time.Time

			serviceFound := false

			for _, record := range reportRecords {
				if strings.Contains(record.Name, dependent) {
					serviceStartTime = record.Started
					serviceFound = true

					break
				}
			}

			if !serviceFound {
				t.Logf("Warning: Service unit not found in report. Cannot verify execution order.")
			} else {
				// Verify each dependency started before service
				for _, depName := range dependencies {
					depFound := false

					for _, record := range reportRecords {
						if strings.Contains(record.Name, depName) {
							depFound = true

							if record.Started.After(serviceStartTime) {
								t.Errorf("Execution order violation: %s started at %v, which is after service started at %v", depName, record.Started, serviceStartTime)
							} else {
								t.Logf("Verified: %s started at %v (before service at %v)", depName, record.Started, serviceStartTime)
							}

							break
						}
					}

					if !depFound {
						t.Logf("Warning: Dependency %s not found in report.", depName)
					}
				}
			}
		}
	} else {
		t.Logf("Warning: Report file not found at %s. Skipping execution order verification.", reportFile)
	}
}

// parseReportFile parses a JSON report file and returns the list of unit names.
func parseReportFile(reportFilePath string) ([]string, error) {
	content, err := os.ReadFile(reportFilePath)
	if err != nil {
		return nil, err
	}

	// Try parsing as JSON first
	var jsonRecords []map[string]interface{}
	if unmarshalErr := json.Unmarshal(content, &jsonRecords); unmarshalErr == nil {
		units := make([]string, 0, len(jsonRecords))
		for _, record := range jsonRecords {
			if name, ok := record["Name"].(string); ok {
				units = append(units, name)
			}
		}

		return units, nil
	}

	// If JSON parsing fails, try CSV
	file, err := os.Open(reportFilePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	reader := csv.NewReader(file)

	csvRecords, err := reader.ReadAll()
	if err != nil {
		return nil, err
	}

	if len(csvRecords) < 2 {
		return []string{}, nil
	}

	// Find the Name column index
	header := csvRecords[0]
	nameIndex := -1

	for i, col := range header {
		if col == "Name" {
			nameIndex = i
			break
		}
	}

	if nameIndex == -1 {
		return nil, errors.New("Name column not found in CSV report")
	}

	// Extract unit names from CSV records
	units := make([]string, 0, len(csvRecords)-1)
	for _, record := range csvRecords[1:] {
		if nameIndex < len(record) {
			units = append(units, record[nameIndex])
		}
	}

	return units, nil
}

// reportRecord represents a record from the run report with timestamps.
type reportRecord struct {
	Name    string
	Started time.Time
	Ended   time.Time
	Result  string
}

// parseReportFileWithTimestamps parses a JSON or CSV report file and returns records with timestamps.
func parseReportFileWithTimestamps(reportFilePath string) ([]reportRecord, error) {
	content, err := os.ReadFile(reportFilePath)
	if err != nil {
		return nil, err
	}

	// Try parsing as JSON first
	var jsonRecords []map[string]interface{}
	if err = json.Unmarshal(content, &jsonRecords); err == nil {
		records := make([]reportRecord, 0, len(jsonRecords))
		for _, record := range jsonRecords {
			r := reportRecord{}
			if name, ok := record["Name"].(string); ok {
				r.Name = name
			}

			if startedStr, ok := record["Started"].(string); ok {
				if t, parseErr := time.Parse(time.RFC3339, startedStr); parseErr == nil {
					r.Started = t
				}
			}

			if endedStr, ok := record["Ended"].(string); ok {
				if t, parseErr := time.Parse(time.RFC3339, endedStr); parseErr == nil {
					r.Ended = t
				}
			}

			if result, ok := record["Result"].(string); ok {
				r.Result = result
			}

			records = append(records, r)
		}

		return records, nil
	}

	// If JSON parsing fails, try CSV
	file, err := os.Open(reportFilePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	reader := csv.NewReader(file)

	csvRecords, err := reader.ReadAll()
	if err != nil {
		return nil, err
	}

	if len(csvRecords) < 2 {
		return []reportRecord{}, nil
	}

	// Find column indices
	header := csvRecords[0]
	nameIndex := -1
	startedIndex := -1
	endedIndex := -1
	resultIndex := -1

	for i, col := range header {
		switch col {
		case "Name":
			nameIndex = i
		case "Started":
			startedIndex = i
		case "Ended":
			endedIndex = i
		case "Result":
			resultIndex = i
		}
	}

	if nameIndex == -1 {
		return nil, errors.New("Name column not found in CSV report")
	}

	// Extract records from CSV
	records := make([]reportRecord, 0, len(csvRecords)-1)
	for _, row := range csvRecords[1:] {
		r := reportRecord{}
		if nameIndex < len(row) {
			r.Name = row[nameIndex]
		}

		if startedIndex >= 0 && startedIndex < len(row) && row[startedIndex] != "" {
			if t, err := time.Parse(time.RFC3339, row[startedIndex]); err == nil {
				r.Started = t
			}
		}

		if endedIndex >= 0 && endedIndex < len(row) && row[endedIndex] != "" {
			if t, err := time.Parse(time.RFC3339, row[endedIndex]); err == nil {
				r.Ended = t
			}
		}

		if resultIndex >= 0 && resultIndex < len(row) {
			r.Result = row[resultIndex]
		}

		records = append(records, r)
	}

	return records, nil
}
