package testing

import (
	"context"
	"testing"

	"github.com/gophercloud/gophercloud/v2/openstack/compute/v2/keypairs"
	"github.com/gophercloud/gophercloud/v2/pagination"
	th "github.com/gophercloud/gophercloud/v2/testhelper"
	"github.com/gophercloud/gophercloud/v2/testhelper/client"
)

func TestList(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleListSuccessfully(t, fakeServer)

	count := 0
	err := keypairs.List(client.ServiceClient(fakeServer), nil).EachPage(context.TODO(), func(_ context.Context, page pagination.Page) (bool, error) {
		count++
		actual, err := keypairs.ExtractKeyPairs(page)
		th.AssertNoErr(t, err)
		th.CheckDeepEquals(t, ExpectedKeyPairSlice, actual)

		return true, nil
	})
	th.AssertNoErr(t, err)
	th.CheckEquals(t, 1, count)
}

func TestCreate(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleCreateSuccessfully(t, fakeServer)

	actual, err := keypairs.Create(context.TODO(), client.ServiceClient(fakeServer), keypairs.CreateOpts{
		Name: "createdkey",
	}).Extract()
	th.AssertNoErr(t, err)
	th.CheckDeepEquals(t, &CreatedKeyPair, actual)
}

func TestCreateOtherUser(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleCreateSuccessfullyOtherUser(t, fakeServer)

	actual, err := keypairs.Create(context.TODO(), client.ServiceClient(fakeServer), keypairs.CreateOpts{
		Name:   "createdkey",
		UserID: "fake2",
	}).Extract()
	th.AssertNoErr(t, err)
	th.CheckDeepEquals(t, &CreatedKeyPairOtherUser, actual)
}

func TestImport(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleImportSuccessfully(t, fakeServer)

	actual, err := keypairs.Create(context.TODO(), client.ServiceClient(fakeServer), keypairs.CreateOpts{
		Name:      "importedkey",
		PublicKey: "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAgQDx8nkQv/zgGgB4rMYmIf+6A4l6Rr+o/6lHBQdW5aYd44bd8JttDCE/F/pNRr0lRE+PiqSPO8nDPHw0010JeMH9gYgnnFlyY3/OcJ02RhIPyyxYpv9FhY+2YiUkpwFOcLImyrxEsYXpD/0d3ac30bNH6Sw9JD9UZHYcpSxsIbECHw== Generated by Nova",
	}).Extract()
	th.AssertNoErr(t, err)
	th.CheckDeepEquals(t, &ImportedKeyPair, actual)
}

func TestGet(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleGetSuccessfully(t, fakeServer)

	actual, err := keypairs.Get(context.TODO(), client.ServiceClient(fakeServer), "firstkey", nil).Extract()
	th.AssertNoErr(t, err)
	th.CheckDeepEquals(t, &FirstKeyPair, actual)
}

func TestGetOtherUser(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleGetSuccessfully(t, fakeServer)

	getOpts := keypairs.GetOpts{
		UserID: "fake2",
	}

	actual, err := keypairs.Get(context.TODO(), client.ServiceClient(fakeServer), "firstkey", getOpts).Extract()
	th.AssertNoErr(t, err)
	th.CheckDeepEquals(t, &FirstKeyPairOtherUser, actual)
}

func TestDelete(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleDeleteSuccessfully(t, fakeServer)

	err := keypairs.Delete(context.TODO(), client.ServiceClient(fakeServer), "deletedkey", nil).ExtractErr()
	th.AssertNoErr(t, err)
}

func TestDeleteOtherUser(t *testing.T) {
	fakeServer := th.SetupHTTP()
	defer fakeServer.Teardown()
	HandleDeleteSuccessfullyOtherUser(t, fakeServer)

	deleteOpts := keypairs.DeleteOpts{
		UserID: "fake2",
	}

	err := keypairs.Delete(context.TODO(), client.ServiceClient(fakeServer), "deletedkey", deleteOpts).ExtractErr()
	th.AssertNoErr(t, err)
}
