#!/usr/bin/env bash
# Copyright 2023 The Go Authors. All rights reserved.
# Use of this source code is governed by a BSD-style
# license that can be found in the LICENSE file.

# This script copies this directory to golang.org/x/exp/trace.
# Just point it at a Go commit or a local Go checkout.

set -e

if [ "$#" -ne 1 ]; then
	echo 'gen.bash expects one argument: a go.googlesource.com/go commit hash to generate the package from or a path to a Go checkout'
	exit 1
fi

# Determine the source.
if [ -d $1 ]; then
	echo "assuming Go checkout at $1..."

	# Select the Go checkout.
	GODIR=$1
else
	echo "using $1 as a commit hash..."

	# Check out Go.
	TMP=$(mktemp -d)
	git -C $TMP clone https://go.googlesource.com/go
	git -C $TMP/go checkout $1
	GODIR=$TMP/go
fi

# Define src and dst.
SRC=$GODIR/src/internal/trace
DST=$(dirname $0)

# Copy.
rsync -av --delete $SRC/ $DST

# Remove the trace_test.go file and the testprogs it invokes.
# This really tests the tracer, it's not necessary to it bring along
# The trace tests are also problematic because they fail to run on
# Go versions before tip. The testprog directory is problematic because
# of //go:build ignore, so we'd have to complicate the logic below to
# support it.
rm $DST/trace_test.go
rm -r $DST/testdata/testprog

# Remove the tracev1 testdata to avoid checking in new binary files.
# Remove tracev1_test.go and internal/tracev1/parser_test.go because
# they fail without this data.
rm -r $DST/internal/tracev1/testdata
rm $DST/tracev1_test.go
rm $DST/internal/tracev1/parser_test.go

# Remove files that are only pertinent to cmd/trace.
rm $DST/export_test.go
rm $DST/gc*.go
rm $DST/mud*.go
rm $DST/summary*.go
rm -r $DST/traceviewer

# Remove mktests.go because its a //go:build ignore file, so it would
# complicate the logic below. This codebase isn't the source of truth
# anyway.
rm $DST/testdata/mktests.go

# Make some packages internal.
mv $DST/raw $DST/internal/raw
mv $DST/tracev2 $DST/internal/tracev2
mv $DST/version $DST/internal/version
mv $DST/testtrace $DST/internal/testtrace

# Fix up import paths.
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's internal/trace golang.org/x/exp/trace '
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's golang.org/x/exp/trace/raw golang.org/x/exp/trace/internal/raw '
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's golang.org/x/exp/trace/tracev2 golang.org/x/exp/trace/internal/tracev2 '
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's golang.org/x/exp/trace/version golang.org/x/exp/trace/internal/version '
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's golang.org/x/exp/trace/testtrace golang.org/x/exp/trace/internal/testtrace '
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e 's internal/txtar golang.org/x/tools/txtar '

# Add build tag for Go 1.21 and generated code comment.
find $DST -name '*.go' | xargs -- sed -i'.tmp' -e '/LICENSE file./a \
\
// Code generated by "gen.bash" from internal/trace; DO NOT EDIT.\
\
//go:build go1.23'

# Format the files.
find $DST -name '*.go' | xargs -- gofmt -w -s

# Delete sed backups
find $DST -name '*.go.tmp' -delete

# Restore known files.
git checkout gen.bash flightrecorder.go flightrecorder_test.go
