package users

import (
	"encoding/json"
	"fmt"
	"strconv"
	"time"

	"github.com/gophercloud/gophercloud/v2"
	"github.com/gophercloud/gophercloud/v2/pagination"
)

// User represents a User in the OpenStack Identity Service.
type User struct {
	// DefaultProjectID is the ID of the default project of the user.
	DefaultProjectID string `json:"default_project_id"`

	// Description is the description of the user.
	Description string `json:"description"`

	// DomainID is the domain ID the user belongs to.
	DomainID string `json:"domain_id"`

	// Enabled is whether or not the user is enabled.
	Enabled bool `json:"-"`

	// Extra is a collection of miscellaneous key/values.
	Extra map[string]any `json:"-"`

	// ID is the unique ID of the user.
	ID string `json:"id"`

	// Links contains referencing links to the user.
	Links map[string]any `json:"links"`

	// Name is the name of the user.
	Name string `json:"name"`

	// Options are a set of defined options of the user.
	Options map[string]any `json:"options"`

	// PasswordExpiresAt is the timestamp when the user's password expires.
	PasswordExpiresAt time.Time `json:"-"`
}

func (r *User) UnmarshalJSON(b []byte) error {
	type tmp User
	var s struct {
		tmp
		Enabled           any                             `json:"enabled"`
		Extra             map[string]any                  `json:"extra"`
		PasswordExpiresAt gophercloud.JSONRFC3339MilliNoZ `json:"password_expires_at"`
	}
	err := json.Unmarshal(b, &s)
	if err != nil {
		return err
	}
	*r = User(s.tmp)

	r.PasswordExpiresAt = time.Time(s.PasswordExpiresAt)

	switch t := s.Enabled.(type) {
	case nil:
		r.Enabled = false
	case bool:
		r.Enabled = t
	case string:
		r.Enabled, err = strconv.ParseBool(t)
		if err != nil {
			return fmt.Errorf("Failed to parse Enabled %q: %v", t, err)
		}
	default:
		return fmt.Errorf("Unknown type for Enabled: %T (value: %v)", t, t)
	}

	// Collect other fields and bundle them into Extra
	// but only if a field titled "extra" wasn't sent.
	if s.Extra != nil {
		r.Extra = s.Extra
	} else {
		var result any
		err := json.Unmarshal(b, &result)
		if err != nil {
			return err
		}
		if resultMap, ok := result.(map[string]any); ok {
			delete(resultMap, "password_expires_at")
			r.Extra = gophercloud.RemainingKeys(User{}, resultMap)
		}
	}

	return err
}

type userResult struct {
	gophercloud.Result
}

// GetResult is the response from a Get operation. Call its Extract method
// to interpret it as a User.
type GetResult struct {
	userResult
}

// CreateResult is the response from a Create operation. Call its Extract method
// to interpret it as a User.
type CreateResult struct {
	userResult
}

// UpdateResult is the response from an Update operation. Call its Extract
// method to interpret it as a User.
type UpdateResult struct {
	userResult
}

// ChangePasswordResult is the response from a ChangePassword operation. Call its
// ExtractErr method to determine if the request succeeded or failed.
type ChangePasswordResult struct {
	gophercloud.ErrResult
}

// DeleteResult is the response from a Delete operation. Call its ExtractErr to
// determine if the request succeeded or failed.
type DeleteResult struct {
	gophercloud.ErrResult
}

// AddToGroupResult is the response from a AddToGroup operation. Call its
// ExtractErr method to determine if the request succeeded or failed.
type AddToGroupResult struct {
	gophercloud.ErrResult
}

// IsMemberOfGroupResult is the response from a IsMemberOfGroup operation. Call its
// Extract method to determine if the request succeeded or failed.
type IsMemberOfGroupResult struct {
	isMember bool
	gophercloud.Result
}

// RemoveFromGroupResult is the response from a RemoveFromGroup operation. Call its
// ExtractErr method to determine if the request succeeded or failed.
type RemoveFromGroupResult struct {
	gophercloud.ErrResult
}

// UserPage is a single page of User results.
type UserPage struct {
	pagination.LinkedPageBase
}

// IsEmpty determines whether or not a UserPage contains any results.
func (r UserPage) IsEmpty() (bool, error) {
	if r.StatusCode == 204 {
		return true, nil
	}

	users, err := ExtractUsers(r)
	return len(users) == 0, err
}

// NextPageURL extracts the "next" link from the links section of the result.
func (r UserPage) NextPageURL() (string, error) {
	var s struct {
		Links struct {
			Next     string `json:"next"`
			Previous string `json:"previous"`
		} `json:"links"`
	}
	err := r.ExtractInto(&s)
	if err != nil {
		return "", err
	}
	return s.Links.Next, err
}

// ExtractUsers returns a slice of Users contained in a single page of results.
func ExtractUsers(r pagination.Page) ([]User, error) {
	var s struct {
		Users []User `json:"users"`
	}
	err := (r.(UserPage)).ExtractInto(&s)
	return s.Users, err
}

// Extract interprets any user results as a User.
func (r userResult) Extract() (*User, error) {
	var s struct {
		User *User `json:"user"`
	}
	err := r.ExtractInto(&s)
	return s.User, err
}

// Extract extracts IsMemberOfGroupResult as bool and error values
func (r IsMemberOfGroupResult) Extract() (bool, error) {
	return r.isMember, r.Err
}
