// Copyright 2020 The Gitea Authors. All rights reserved.
// SPDX-License-Identifier: MIT

package pulls

import (
	stdctx "context"

	"code.gitea.io/sdk/gitea"
	"code.gitea.io/tea/cmd/flags"
	"code.gitea.io/tea/modules/context"
	"code.gitea.io/tea/modules/print"
	"github.com/urfave/cli/v3"
)

var pullFieldsFlag = flags.FieldsFlag(print.PullFields, []string{
	"index", "title", "state", "author", "milestone", "updated", "labels",
})

// CmdPullsList represents a sub command of issues to list pulls
var CmdPullsList = cli.Command{
	Name:        "list",
	Aliases:     []string{"ls"},
	Usage:       "List pull requests of the repository",
	Description: `List pull requests of the repository`,
	ArgsUsage:   " ", // command does not accept arguments
	Action:      RunPullsList,
	Flags:       append([]cli.Flag{pullFieldsFlag}, flags.PRListingFlags...),
}

// RunPullsList return list of pulls
func RunPullsList(_ stdctx.Context, cmd *cli.Command) error {
	ctx := context.InitCommand(cmd)
	ctx.Ensure(context.CtxRequirement{RemoteRepo: true})

	state := gitea.StateOpen
	switch ctx.String("state") {
	case "all":
		state = gitea.StateAll
	case "open":
		state = gitea.StateOpen
	case "closed":
		state = gitea.StateClosed
	}

	prs, _, err := ctx.Login.Client().ListRepoPullRequests(ctx.Owner, ctx.Repo, gitea.ListPullRequestsOptions{
		State: state,
	})

	if err != nil {
		return err
	}

	fields, err := pullFieldsFlag.GetValues(cmd)
	if err != nil {
		return err
	}

	print.PullsList(prs, ctx.Output, fields)
	return nil
}
