//
// Copyright 2021, Sander van Harmelen
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package gitlab

import (
	"bytes"
	"fmt"
	"net/http"
	"time"
)

type (
	SnippetsServiceInterface interface {
		ListSnippets(opt *ListSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error)
		GetSnippet(snippet int64, options ...RequestOptionFunc) (*Snippet, *Response, error)
		SnippetContent(snippet int64, options ...RequestOptionFunc) ([]byte, *Response, error)
		SnippetFileContent(snippet int64, ref, filename string, options ...RequestOptionFunc) ([]byte, *Response, error)
		CreateSnippet(opt *CreateSnippetOptions, options ...RequestOptionFunc) (*Snippet, *Response, error)
		UpdateSnippet(snippet int64, opt *UpdateSnippetOptions, options ...RequestOptionFunc) (*Snippet, *Response, error)
		DeleteSnippet(snippet int64, options ...RequestOptionFunc) (*Response, error)
		ExploreSnippets(opt *ExploreSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error)
		ListAllSnippets(opt *ListAllSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error)
	}

	// SnippetsService handles communication with the snippets
	// related methods of the GitLab API.
	//
	// GitLab API docs: https://docs.gitlab.com/api/snippets/
	SnippetsService struct {
		client *Client
	}
)

var _ SnippetsServiceInterface = (*SnippetsService)(nil)

// Snippet represents a GitLab snippet.
//
// GitLab API docs: https://docs.gitlab.com/api/snippets/
type Snippet struct {
	ID                int64         `json:"id"`
	Title             string        `json:"title"`
	FileName          string        `json:"file_name"`
	Description       string        `json:"description"`
	Visibility        string        `json:"visibility"`
	Author            SnippetAuthor `json:"author"`
	UpdatedAt         *time.Time    `json:"updated_at"`
	CreatedAt         *time.Time    `json:"created_at"`
	ProjectID         int64         `json:"project_id"`
	WebURL            string        `json:"web_url"`
	RawURL            string        `json:"raw_url"`
	Files             []SnippetFile `json:"files"`
	RepositoryStorage string        `json:"repository_storage"`
}

func (s Snippet) String() string {
	return Stringify(s)
}

// SnippetAuthor represents a GitLab snippet author.
//
// GitLab API docs: https://docs.gitlab.com/api/snippets/
type SnippetAuthor struct {
	ID        int64      `json:"id"`
	Username  string     `json:"username"`
	Email     string     `json:"email"`
	Name      string     `json:"name"`
	State     string     `json:"state"`
	CreatedAt *time.Time `json:"created_at"`
}

func (a SnippetAuthor) String() string {
	return Stringify(a)
}

// SnippetFile represents a GitLab snippet file.
//
// GitLab API docs: https://docs.gitlab.com/api/snippets/
type SnippetFile struct {
	Path   string `json:"path"`
	RawURL string `json:"raw_url"`
}

func (f SnippetFile) String() string {
	return Stringify(f)
}

// ListSnippetsOptions represents the available ListSnippets() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-snippets-for-current-user
type ListSnippetsOptions struct {
	ListOptions
}

// ListSnippets gets a list of snippets.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-snippets-for-current-user
func (s *SnippetsService) ListSnippets(opt *ListSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error) {
	res, resp, err := do[[]*Snippet](s.client,
		withMethod(http.MethodGet),
		withPath("snippets"),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}

// GetSnippet gets a single snippet
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#get-a-single-snippet
func (s *SnippetsService) GetSnippet(snippet int64, options ...RequestOptionFunc) (*Snippet, *Response, error) {
	res, resp, err := do[*Snippet](s.client,
		withMethod(http.MethodGet),
		withPath("snippets/%d", snippet),
		withAPIOpts(nil),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}

// SnippetContent gets a single snippet’s raw contents.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#single-snippet-contents
func (s *SnippetsService) SnippetContent(snippet int64, options ...RequestOptionFunc) ([]byte, *Response, error) {
	u := fmt.Sprintf("snippets/%d/raw", snippet)

	req, err := s.client.NewRequest(http.MethodGet, u, nil, options)
	if err != nil {
		return nil, nil, err
	}

	var b bytes.Buffer
	resp, err := s.client.Do(req, &b)
	if err != nil {
		return nil, resp, err
	}

	return b.Bytes(), resp, err
}

// SnippetFileContent returns the raw file content as plain text.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#snippet-repository-file-content
func (s *SnippetsService) SnippetFileContent(snippet int64, ref, filename string, options ...RequestOptionFunc) ([]byte, *Response, error) {
	filepath := PathEscape(filename)
	u := fmt.Sprintf("snippets/%d/files/%s/%s/raw", snippet, ref, filepath)

	req, err := s.client.NewRequest(http.MethodGet, u, nil, options)
	if err != nil {
		return nil, nil, err
	}

	var b bytes.Buffer
	resp, err := s.client.Do(req, &b)
	if err != nil {
		return nil, resp, err
	}

	return b.Bytes(), resp, err
}

// CreateSnippetFileOptions represents the create snippet file options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#create-new-snippet
type CreateSnippetFileOptions struct {
	FilePath *string `url:"file_path,omitempty" json:"file_path,omitempty"`
	Content  *string `url:"content,omitempty" json:"content,omitempty"`
}

// CreateSnippetOptions represents the available CreateSnippet() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#create-new-snippet
type CreateSnippetOptions struct {
	Title       *string                      `url:"title,omitempty" json:"title,omitempty"`
	FileName    *string                      `url:"file_name,omitempty" json:"file_name,omitempty"`
	Description *string                      `url:"description,omitempty" json:"description,omitempty"`
	Content     *string                      `url:"content,omitempty" json:"content,omitempty"`
	Visibility  *VisibilityValue             `url:"visibility,omitempty" json:"visibility,omitempty"`
	Files       *[]*CreateSnippetFileOptions `url:"files,omitempty" json:"files,omitempty"`
}

// CreateSnippet creates a new snippet. The user must have permission
// to create new snippets.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#create-new-snippet
func (s *SnippetsService) CreateSnippet(opt *CreateSnippetOptions, options ...RequestOptionFunc) (*Snippet, *Response, error) {
	res, resp, err := do[*Snippet](s.client,
		withMethod(http.MethodPost),
		withPath("snippets"),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}

// UpdateSnippetFileOptions represents the update snippet file options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#update-snippet
type UpdateSnippetFileOptions struct {
	Action       *string `url:"action,omitempty" json:"action,omitempty"`
	FilePath     *string `url:"file_path,omitempty" json:"file_path,omitempty"`
	Content      *string `url:"content,omitempty" json:"content,omitempty"`
	PreviousPath *string `url:"previous_path,omitempty" json:"previous_path,omitempty"`
}

// UpdateSnippetOptions represents the available UpdateSnippet() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#update-snippet
type UpdateSnippetOptions struct {
	Title       *string                      `url:"title,omitempty" json:"title,omitempty"`
	FileName    *string                      `url:"file_name,omitempty" json:"file_name,omitempty"`
	Description *string                      `url:"description,omitempty" json:"description,omitempty"`
	Content     *string                      `url:"content,omitempty" json:"content,omitempty"`
	Visibility  *VisibilityValue             `url:"visibility,omitempty" json:"visibility,omitempty"`
	Files       *[]*UpdateSnippetFileOptions `url:"files,omitempty" json:"files,omitempty"`
}

// UpdateSnippet updates an existing snippet. The user must have
// permission to change an existing snippet.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#update-snippet
func (s *SnippetsService) UpdateSnippet(snippet int64, opt *UpdateSnippetOptions, options ...RequestOptionFunc) (*Snippet, *Response, error) {
	res, resp, err := do[*Snippet](s.client,
		withMethod(http.MethodPut),
		withPath("snippets/%d", snippet),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}

// DeleteSnippet deletes an existing snippet. This is an idempotent
// function and deleting a non-existent snippet still returns a 200 OK status
// code.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#delete-snippet
func (s *SnippetsService) DeleteSnippet(snippet int64, options ...RequestOptionFunc) (*Response, error) {
	_, resp, err := do[none](s.client,
		withMethod(http.MethodDelete),
		withPath("snippets/%d", snippet),
		withAPIOpts(nil),
		withRequestOpts(options...),
	)
	return resp, err
}

// ExploreSnippetsOptions represents the available ExploreSnippets() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-public-snippets
type ExploreSnippetsOptions struct {
	ListOptions
}

// ExploreSnippets gets the list of public snippets.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-public-snippets
func (s *SnippetsService) ExploreSnippets(opt *ExploreSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error) {
	res, resp, err := do[[]*Snippet](s.client,
		withMethod(http.MethodGet),
		withPath("snippets/public"),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}

// ListAllSnippetsOptions represents the available ListAllSnippets() options.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-snippets
type ListAllSnippetsOptions struct {
	ListOptions
	CreatedAfter      *ISOTime `url:"created_after,omitempty" json:"created_after,omitempty"`
	CreatedBefore     *ISOTime `url:"created_before,omitempty" json:"created_before,omitempty"`
	RepositoryStorage *string  `url:"repository_storage,omitempty" json:"repository_storage,omitempty"`
}

// ListAllSnippets gets all snippets the current user has access to.
//
// GitLab API docs:
// https://docs.gitlab.com/api/snippets/#list-all-snippets
func (s *SnippetsService) ListAllSnippets(opt *ListAllSnippetsOptions, options ...RequestOptionFunc) ([]*Snippet, *Response, error) {
	res, resp, err := do[[]*Snippet](s.client,
		withMethod(http.MethodGet),
		withPath("snippets/all"),
		withAPIOpts(opt),
		withRequestOpts(options...),
	)
	if err != nil {
		return nil, resp, err
	}
	return res, resp, nil
}
