package appsec

import (
	"context"
	"encoding/json"
	"errors"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/akamai/AkamaiOPEN-edgegrid-golang/v11/pkg/session"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAppSec_GetMalwarePolicy(t *testing.T) {

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicyResponse.json"))
	result := MalwarePolicyResponse{}
	err := json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           GetMalwarePolicyRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *MalwarePolicyResponse
		withError        error
	}{
		"200 OK": {
			params: GetMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
			},
			responseStatus:   http.StatusOK,
			responseBody:     respData,
			expectedPath:     "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
			expectedResponse: &result,
		},
		"500 internal server error": {
			params: GetMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
			{
				"type": "internal_error",
				"title": "Internal Server Error",
				"detail": "Error fetching match target"
			}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error fetching match target",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, test.expectedPath, r.URL.String())
				assert.Equal(t, http.MethodGet, r.Method)
				w.WriteHeader(test.responseStatus)
				_, err := w.Write([]byte(test.responseBody))
				assert.NoError(t, err)
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.GetMalwarePolicy(context.Background(), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}

func TestAppSec_ListMalwareProtectionPolicies(t *testing.T) {

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicies.json"))
	result := MalwarePoliciesResponse{}
	err := json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           GetMalwarePoliciesRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *MalwarePoliciesResponse
		withError        error
		headers          http.Header
	}{
		"200 OK": {
			params: GetMalwarePoliciesRequest{
				ConfigID:      43253,
				ConfigVersion: 15,
			},
			headers: http.Header{
				"Content-Type": []string{"application/json"},
			},
			responseStatus:   http.StatusOK,
			responseBody:     string(respData),
			expectedPath:     "/appsec/v1/configs/43253/versions/15/malware-policies",
			expectedResponse: &result,
		},
		"500 internal server error": {
			params: GetMalwarePoliciesRequest{
				ConfigID:      43253,
				ConfigVersion: 15,
			},
			headers:        http.Header{},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
{
    "type": "internal_error",
    "title": "Internal Server Error",
    "detail": "Error fetching propertys",
    "status": 500
}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error fetching propertys",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, test.expectedPath, r.URL.String())
				assert.Equal(t, http.MethodGet, r.Method)
				w.WriteHeader(test.responseStatus)
				_, err := w.Write([]byte(test.responseBody))
				assert.NoError(t, err)
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.GetMalwarePolicies(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers),
				),
				test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}

// Test Create MalwarePolicy
func TestAppSec_CreateMalwarePolicy(t *testing.T) {

	reqData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicyDescription.json"))
	req := MalwarePolicyBody{}
	err := json.Unmarshal([]byte(reqData), &req)
	require.NoError(t, err)

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicyResponse.json"))
	result := MalwarePolicyResponse{}
	err = json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           CreateMalwarePolicyRequest
		prop             *CreateMalwarePolicyRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *MalwarePolicyResponse
		withError        error
		headers          http.Header
	}{
		"201 Created": {
			params: CreateMalwarePolicyRequest{
				ConfigID:      43253,
				ConfigVersion: 15,
				Policy:        &req,
			},
			headers: http.Header{
				"Content-Type": []string{"application/json;charset=UTF-8"},
			},
			responseStatus:   http.StatusCreated,
			responseBody:     respData,
			expectedResponse: &result,
			expectedPath:     "/appsec/v1/configs/43253/versions/15/malware-policies",
		},
		"500 internal server error": {
			params: CreateMalwarePolicyRequest{
				ConfigID:      43253,
				ConfigVersion: 15,
				Policy:        &req,
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
			{
				"type": "internal_error",
				"title": "Internal Server Error",
				"detail": "Error creating domain"
			}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error creating domain",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, http.MethodPost, r.Method)
				w.WriteHeader(test.responseStatus)
				if len(test.responseBody) > 0 {
					_, err := w.Write([]byte(test.responseBody))
					assert.NoError(t, err)
				}
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.CreateMalwarePolicy(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers)), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}

// Test Update MalwarePolicy
func TestAppSec_UpdateMalwarePolicy(t *testing.T) {

	reqData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicyDescription.json"))
	req := MalwarePolicyBody{}
	err := json.Unmarshal([]byte(reqData), &req)
	require.NoError(t, err)

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwarePolicy/MalwarePolicyResponse.json"))
	result := MalwarePolicyResponse{}
	err = json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           UpdateMalwarePolicyRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *MalwarePolicyResponse
		withError        error
		headers          http.Header
		logFilename      bool
	}{
		"200 Success": {
			params: UpdateMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
				Policy:          &req,
			},
			headers: http.Header{
				"Content-Type": []string{"application/json;charset=UTF-8"},
			},
			responseStatus:   http.StatusCreated,
			responseBody:     respData,
			expectedResponse: &result,
			expectedPath:     "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
			logFilename:      true,
		},
		"500 internal server error": {
			params: UpdateMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
				Policy:          &req,
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
			{
				"type": "internal_error",
				"title": "Internal Server Error",
				"detail": "Error creating zone"
			}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error creating zone",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, http.MethodPut, r.Method)
				w.WriteHeader(test.responseStatus)
				if len(test.responseBody) > 0 {
					_, err := w.Write([]byte(test.responseBody))
					assert.NoError(t, err)
				}
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.UpdateMalwarePolicy(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers)), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}

// Test Remove MalwarePolicy
func TestAppSec_RemoveMalwarePolicy(t *testing.T) {

	tests := map[string]struct {
		params         RemoveMalwarePolicyRequest
		responseStatus int
		responseBody   string
		expectedPath   string
		withError      error
		headers        http.Header
	}{
		"200 Success": {
			params: RemoveMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
			},
			headers: http.Header{
				"Content-Type": []string{"application/json;charset=UTF-8"},
			},
			responseStatus: http.StatusOK,
			expectedPath:   "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
		},
		"500 internal server error": {
			params: RemoveMalwarePolicyRequest{
				ConfigID:        43253,
				ConfigVersion:   15,
				MalwarePolicyID: 134644,
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: `
			{
				"type": "internal_error",
				"title": "Internal Server Error",
				"detail": "Error deleting match target"
			}`,
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies/134644",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error deleting match target",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, http.MethodDelete, r.Method)
				w.WriteHeader(test.responseStatus)
				if len(test.responseBody) > 0 {
					_, err := w.Write([]byte(test.responseBody))
					assert.NoError(t, err)
				}
			}))
			client := mockAPIClient(t, mockServer)
			err := client.RemoveMalwarePolicy(
				session.ContextWithOptions(
					context.Background(),
					session.WithContextHeaders(test.headers)), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
		})
	}
}
