// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package samreg

import (
	"slices"
	"strings"
	"testing"
)

func TestDomainFDeriveSyskey(t *testing.T) {
	tests := []struct {
		name        string
		syskey      []byte
		buffer      []byte
		want        []byte
		wantErr     bool
		wantErrText string
	}{
		{
			name:   "derivation_RC4_succeeds",
			syskey: []byte("\x88\x93\xae\x93\x45\x13\xbd\xdd\x25\x47\x35\x16\x3e\x9d\x33\x00"),
			buffer: []byte("\x02\x00\x01\x00\x00\x00\x00\x00\x40\x15\x3b\x97\x46\x9f\xce\x01\x26\x00\x00\x00\x00\x00\x00\x00\x00\x80\xa6\x0a\xff\xde\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\xcc\x1d\xcf\xfb\xff\xff\xff\x00\xcc\x1d\xcf\xfb\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\xe9\x03\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x01\x00\x00\x00\x38\x00\x00\x00\x23\x7e\xe9\x12\xa7\x34\xbf\x93\x18\x6e\xaa\xc1\x83\x07\x59\xa1\xd6\x96\xa6\x99\x6b\xa9\x41\x61\x44\x92\xb0\xfb\xd0\x0a\xe9\xa6\x37\xd6\x7c\xc6\x99\x2b\xc7\x12\xfe\x22\xa0\x17\x71\xce\xd3\xaa\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x38\x00\x00\x00\x3d\xfe\xe0\xd7\x20\xeb\x39\xc1\x44\x1c\x8d\x05\x29\xd6\x83\x47\x92\xa2\x29\x38\xfc\x9e\xa7\x29\xa9\x36\x7d\x4a\xfc\x6c\xe1\xb3\xd3\xac\xd4\xac\xe2\x5b\xab\xf9\xf8\x3f\x09\xe1\x91\x1a\x7d\xda\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00"),
			want:   []byte("\x3d\x21\x2c\xe8\xa2\xda\x83\x43\xbd\xad\x1e\xf2\xcf\xb6\xb3\x1c"),
		},
		{
			name:   "derivation_AES_succeeds",
			syskey: []byte("\x95\x60\xbf\x6a\x47\xe0\xc9\x02\xb0\x82\x45\xf1\x96\xd5\x4e\x4f"),
			buffer: []byte("\x03\x00\x01\x00\x00\x00\x00\x00\x7d\xdc\xe4\xd7\xae\x0a\xdb\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x80\xa6\x0a\xff\xde\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x44\x5f\x9a\xfe\xff\xff\xff\x00\x44\x5f\x9a\xfe\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\xea\x03\x00\x00\x08\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x01\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x02\x00\x00\x00\x70\x00\x00\x00\x30\x00\x00\x00\x20\x00\x00\x00\x69\xcc\x39\x99\xf8\xc2\x32\xee\x75\x6e\xfe\x19\xe5\x52\xfd\x7f\xdf\x24\xbb\x41\xd5\x88\x39\x31\x28\x60\x6e\x96\x0f\x0d\x9c\xc1\x86\x40\xd1\x41\x5b\x0a\x0b\x74\x36\xae\xcd\xcf\xb9\x64\xe1\x1d\xbb\x2c\x3a\x60\xf9\xab\x0e\x36\x02\xe6\x29\x24\x54\xed\xfa\x0e\xf3\xbe\x13\xbd\x24\x0a\x9c\x96\xca\xeb\xcc\x6c\x18\x19\x3e\x57\xb8\x51\x80\x81\x2a\x13\xe3\x1c\x76\x11\xd2\x47\x43\xaa\x10\x12\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x70\x00\x00\x00\x30\x00\x00\x00\x20\x00\x00\x00\x17\xf1\xdf\x69\x49\x14\xce\x2b\xae\xa0\xf6\x6b\x79\x2f\xb3\x90\x9d\x08\x11\xbe\x24\xed\x58\x2a\x18\x60\xb3\x51\x63\x73\xd2\x3d\x88\x7c\xd5\x11\x17\x2c\xdc\x80\xa3\x32\x14\xd3\x4a\x08\xf5\x91\x5e\x47\xc2\xb4\x8f\x09\x4a\x7b\xd6\x37\x1b\xa9\xd1\x9a\x77\x05\xb0\xb7\x4f\xe0\xa8\xeb\x5b\xb6\x82\xfb\x00\x44\xd1\x13\xf6\x31\x57\xf2\x2f\xb8\x61\x67\x60\x4e\x63\x6a\xed\x83\x0c\x72\x55\x69\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00"),
			want:   []byte("\xfc\xde\xe8\x3a\xc6\xc1\x4b\x28\xf5\x26\x50\x1f\xc6\xe8\xbb\xc3"),
		},
		{
			name:        "domainF_too_short_RC4_returns_error",
			buffer:      []byte("A"),
			wantErr:     true,
			wantErrText: errDomainFTooShort.Error(),
		},
		{
			name:        "domainF_too_short_AES_returns_error",
			buffer:      []byte(strings.Repeat("A", fDomStructKeyOffset) + "\x02" + strings.Repeat("A", 32)),
			wantErr:     true,
			wantErrText: errDomainFTooShort.Error(),
		},
		{
			name:        "invalid_RC4_checksum_returns_error",
			syskey:      []byte("\x88\x93\xae\x93\x45\x13\xbd\xdd\x25\x47\x34\x16\x3e\x9d\x33\x00"),
			buffer:      []byte("\x02\x00\x01\x00\x00\x00\x00\x00\x40\x15\x3b\x97\x46\x9f\xce\x01\x26\x00\x00\x00\x00\x00\x00\x00\x00\x80\xa6\x0a\xff\xde\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\xcc\x1d\xcf\xfb\xff\xff\xff\x00\xcc\x1d\xcf\xfb\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\xe9\x03\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x03\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x01\x00\x00\x00\x38\x00\x00\x00\x23\x7e\xe9\x12\xa7\x34\xbf\x93\x18\x6e\xaa\xc1\x83\x07\x59\xa1\xd6\x96\xa6\x99\x6b\xa9\x41\x61\x44\x92\xb0\xfb\xd0\x0a\xe9\xa6\x37\xd6\x7c\xc6\x99\x2b\xc7\x12\xfe\x22\xa0\x17\x71\xce\xd3\xaa\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x38\x00\x00\x00\x3d\xfe\xe0\xd7\x20\xeb\x39\xc1\x44\x1c\x8d\x05\x29\xd6\x83\x47\x92\xa2\x29\x38\xfc\x9e\xa7\x29\xa9\x36\x7d\x4a\xfc\x6c\xe1\xb3\xd3\xac\xd4\xac\xe2\x5b\xab\xf9\xf8\x3f\x09\xe1\x91\x1a\x7d\xda\x00\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00"),
			wantErr:     true,
			wantErrText: errInvalidChecksum.Error(),
		},
		{
			name:        "invalid_revision_returns_error",
			buffer:      []byte(strings.Repeat("A", fDomStructKeyOffset) + "\xFF"),
			wantErr:     true,
			wantErrText: errInvalidRevision.Error(),
		},
		{
			name:        "decode_syskey_RC4_failure_returns_error",
			buffer:      []byte(strings.Repeat("A", fDomStructKeyOffset) + "\x01"),
			wantErr:     true,
			wantErrText: "unexpected EOF",
		},
		{
			name:        "decode_syskey_AES_failure_returns_error",
			buffer:      []byte(strings.Repeat("A", fDomStructKeyOffset) + "\x02"),
			wantErr:     true,
			wantErrText: "unexpected EOF",
		},
		{
			name:        "missing_syskey_AES_returns_error",
			syskey:      []byte(""),
			buffer:      []byte(strings.Repeat("\x00", fDomStructKeyOffset) + "\x02" + strings.Repeat("\x00", 0xCC)),
			wantErr:     true,
			wantErrText: "crypto/aes: invalid key size 0",
		},
	}

	for _, tc := range tests {
		t.Run(tc.name, func(t *testing.T) {
			got, gotErr := newDomainF(tc.buffer).DeriveSyskey(tc.syskey)
			if (gotErr != nil) != tc.wantErr {
				t.Errorf("DeriveSyskey(...): unexpected error: %v", gotErr)
			}

			if tc.wantErr {
				if !strings.Contains(gotErr.Error(), tc.wantErrText) {
					t.Errorf("DeriveSyskey(...): unexpected error, got: %v, want: %v", gotErr, tc.wantErrText)
				}

				return
			}

			if !slices.Equal(got, tc.want) {
				t.Errorf("DeriveSyskey(...): got %x, want %x", got, tc.want)
			}
		})
	}
}
