//go:build fossil

package fossilfs

import (
	"testing"

	"github.com/gopasspw/gopass/internal/backend/storage/fs"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestFossil_Get(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	name := "test"

	fossil.fs.Set(ctx, name, []byte("content"))

	content, err := fossil.Get(ctx, name)
	require.NoError(t, err)
	assert.Equal(t, []byte("content"), content)
}

func TestFossil_Set(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	name := "test"
	value := []byte("content")

	err := fossil.Set(ctx, name, value)
	require.NoError(t, err)
}

func TestFossil_Delete(t *testing.T) {
	t.Skip("needs fossil binary")

	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	name := "test"

	fossil.fs.Set(ctx, name, []byte("content"))

	err := fossil.Delete(ctx, name)
	require.NoError(t, err)
}

func TestFossil_Exists(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	name := "test"

	fossil.fs.Set(ctx, name, []byte("content"))

	exists := fossil.Exists(ctx, name)
	assert.True(t, exists)
}

func TestFossil_List(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	prefix := "test"

	fossil.fs.Set(ctx, "test/foo", []byte("content"))
	fossil.fs.Set(ctx, "test/bar", []byte("content"))
	fossil.fs.Set(ctx, "foo/bar", []byte("content"))

	list, err := fossil.List(ctx, prefix)
	require.NoError(t, err)
	assert.Equal(t, []string{"test/bar", "test/foo"}, list)
}

func TestFossil_IsDir(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	name := "test"

	fossil.fs.Set(ctx, "test/foo", []byte("content"))

	assert.True(t, fossil.IsDir(ctx, name))
}

func TestFossil_Prune(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	prefix := "test"

	fossil.fs.Set(ctx, "test/foo", []byte("content"))
	fossil.fs.Set(ctx, "test/bar", []byte("content"))

	err := fossil.Prune(ctx, prefix)
	require.NoError(t, err)
}

func TestFossil_String(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}

	str := fossil.String()
	assert.Contains(t, str, "fossilfs(")
	assert.Contains(t, str, "path:/path/to/storage")
}

func TestFossil_Path(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}

	path := fossil.Path()
	assert.Equal(t, "/path/to/storage", path)
}

func TestFossil_Fsck(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()

	err := fossil.Fsck(ctx)
	require.NoError(t, err)
}

func TestFossil_Link(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	from := "from"
	to := "to"

	fossil.fs.Set(ctx, "from", []byte("content"))

	err := fossil.Link(ctx, from, to)
	require.NoError(t, err)
}

func TestFossil_Move(t *testing.T) {
	td := t.TempDir()
	fossil := &Fossil{fs: fs.New(td)}
	ctx := t.Context()
	src := "src"
	dst := "dst"
	del := true

	fossil.fs.Set(ctx, "src", []byte("content"))

	err := fossil.Move(ctx, src, dst, del)
	require.NoError(t, err)
}
